#include <avr/io.h>
#include <util/delay.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>

#define KEY _BV(PB0)
#define OSC _BV(PB1)
#define SWITCH _BV(PB2)

enum { IDLE = 0, START = 1, STOP = 10 };

unsigned char txState;
unsigned char txByte;

void init() {
	cli();

	DDRB |= KEY | OSC;
	PORTB = 0;
	
	// timer generates interrupts at ~20 Hz
	TCCR0A = _BV(WGM01);		// CTC mode
	TCCR0B = _BV(CS02) | _BV(CS00);	// Prescale 1024
	OCR0A = 58;			// TOP = 58
	TIMSK0 = _BV(OCIE0A);		// Interrupt at TOP

	// pin change interrupt on the switch line
	DDRB &= ~SWITCH;
	PORTB &= ~SWITCH;		// clear internal pull-up
	PCMSK |= _BV(PCINT2);
	GIMSK |= _BV(PCIE);

	txState = IDLE;
	txByte = 0x5A; // 0101 1010

	sei();
}

ISR(PCINT0_vect) {
	// debounce
	_delay_ms(10);
	if((PINB & SWITCH) && txState == IDLE) {
		GTCCR |= _BV(PSR10);		// clear prescaler
		TCNT0 = 0;			// clear timer
		txState = START;		// start TX
	}
}

ISR(TIM0_COMPA_vect) {
	switch(txState) {
		case IDLE:
			// nothing
			break;
		case START:
			PORTB |= KEY | OSC;
			txState++;
			break;
		case STOP:
			PORTB &= ~(KEY | OSC);
			txState = IDLE;
			break;
		default:
			PORTB = (PORTB & ~KEY) | ((txByte & (1 << (txState - 2)))?(KEY):(0));
			txState++;
			break;
	}
}

int main() {
	init();

	while(1) {
		set_sleep_mode(SLEEP_MODE_IDLE);
		sleep_mode();
	}

	return 0; // unreached
}
