#include <avr/io.h>
#include <util/delay.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>

#define OSCCTL _BV(PB0)		// oscillator control/sense line
#define CARDET _BV(PB1)		// carrier detection signal (drives LED, etc)

#define OUTPUT _BV(PB2)		// output signal

#define MAXCOUNT 3000		// oscillator should start within 3 ms
#define NOISE_AVERAGE 20	// measure and average the noise floor this many times
#define NOISE_MARGIN 70		// margin in us for carrier detection above noise floor

#define HZ_BAUD 58		// baud rate ~20 Hz
#define HZ_FAST 11		// over-sampling rate for start bit detection ~100 Hz
#define BITLEN 50		// bit length in ms

enum { IDLE = 0, START = 1, STOP = 10 };

unsigned char rxState;
unsigned char rxByte;

unsigned int threshold;

unsigned int sample() {
	unsigned int count;

	// tristate/input control pin to allow oscillator to start
	DDRB &= ~OSCCTL;

	// wait for oscillation to build counting micro-seconds (approximately)
	for(count = 0; count < MAXCOUNT; count++) {
		if(PINB & OSCCTL) break;
		_delay_us(1);
	}

	// drive control pin low to stop oscillation
	// we leave the control line low until next sampling
	// to let oscillation die away completely
	PORTB &= ~OSCCTL;
	DDRB |= OSCCTL;	


	// return the sampled signal strength
	// zero would indicate oscillator failed to start and is an error condition
	return (MAXCOUNT - count);
}

void setTimer(char top) {
	// timer generates interrupts at roughly 20 or 100 Hz

	OCR0A = top;			// set TOP

	TCCR0A = _BV(WGM01);		// CTC mode
	TCCR0B = _BV(CS02) | _BV(CS00);	// Prescale 1024
	TIMSK0 = _BV(OCIE0A);		// Interrupt at TOP

	GTCCR |= _BV(PSR10);		// clear prescaler
	TCNT0 = 0;			// clear timer
	TIFR0 = 0;			// clear timer interrupt flags
}

void processByte() {
	// if RX code, toggle OUTPUT
	if(rxByte == 0x5A) PINB |= OUTPUT;

	// this can obviously be much more clever
}

void calibrateThreshold() {
	int i;

	// we setup the carrier detection threshold based on
	// the measured average noise level plus a hard-coded noise margin
	// obviously this is a little fragile and should be improved!

	sample();
	_delay_ms(20);
	threshold = sample();
	for(i = 0; i < NOISE_AVERAGE; i++) {
		_delay_ms(20);
		threshold += sample();
		threshold >>= 1;
	}
	threshold += NOISE_MARGIN;
}

void init() {
	cli();

	PORTB = 0;
	DDRB |= CARDET | OUTPUT;

	calibrateThreshold();

	rxByte = 0;
	rxState = IDLE;

	setTimer(HZ_FAST);

	sei();
}

ISR(TIM0_COMPA_vect) {
	// measure current signal strength
	unsigned int level = sample();
	
	// assert carrier detected if signal above threshold
	PORTB = (PORTB & ~CARDET) | (!level || (level > threshold)?(CARDET):(0));

	switch(rxState) {
		case IDLE:
			if(level > threshold) {
				// start bit seen
				rxState = START;
				_delay_ms(BITLEN/2); // wait until near the middle of start bit
				setTimer(HZ_BAUD); // start sampling at the baud rate
			}
			break;
		case START:
			rxByte = 0;
			rxState++;
			break;
		default:
			// data bit
			rxByte |= ((level > threshold)?(1 << (rxState - 2)):(0));
			rxState++;
			break;
		case STOP:
			processByte();
			setTimer(HZ_FAST);  // resume fast sampling for start bit
			rxState = IDLE;
			break;
	}
}

int main() {
	init();

	while(1) {
		set_sleep_mode(SLEEP_MODE_IDLE);
		sleep_mode();
	}

	return 0; // unreached
}
